/**
 * Implementation of a Mech Heat Tracker as application for the 
 * MechWarrior: Living Legends Mod.
 *
 * @author Michael Barth <mbarth@little-things.de>
 * @created 2.04.2008
 */

#include "StdAfx.h"
#include "MechHeatTrack.h"

using namespace std;

#define CRIT_TEMP_THRESHOLD	100
#define OVERHEAT_THRESHOLD	60
#define TESTING 1		// set to 1 for testing output on the console

CMechHeatTrack::CMechHeatTrack(
		float baseCoolRate, 
		int heatSinkCount, 
		HeatSinkTypes typeOfHeatSinks, 
		int coolantCapacity)
{
	// initialize MechHeatTrack
	baseCoolingFactor = baseCoolRate;
	numberOfHeatSinks = heatSinkCount;
	typeOfHeatSink = typeOfHeatSinks;
	coolantInTank = coolantCapacity;

	// start with cool environment
	overHeat = false;
	criticalTemp = false;
	totalHeat = 0.0f;
	heatsinkDamage = 0.0f;
	environment = IN_TEMPARATE_AIR;
	rateOfHeatChange = 0.0f;
	additionalHeat = 0.0f;
	additionalCooling = 0.0f;
}

void CMechHeatTrack::UpdateCooling()
{
	if (TESTING) {
		cout << "Updating cooling..." << endl;
	}

	// get latest heat rate and add it to the total heat
	// cooling results in a negative heat rate
	rateOfHeatChange = GetRateOfHeatChange();
	totalHeat = totalHeat + rateOfHeatChange;
	if (totalHeat < 0)
		totalHeat = 0;		// don't go below 0
	
	// set overheat and critical temp flags
	if(totalHeat > OVERHEAT_THRESHOLD) 
	{
		overHeat = true;

		if(totalHeat > CRIT_TEMP_THRESHOLD)
			criticalTemp = true;
		else
			criticalTemp = false;	
	}
	else 
	{
		overHeat = false;
	}

	if (TESTING) {
		cout << "New heat level: " << totalHeat << endl;
		cout << "Overheated? " << (overHeat ? "yes" : "no") << " ";
		cout << "Critical Temp? " << (criticalTemp ? "yes" : "no") << "\n\n";
	}
}

void CMechHeatTrack::UpdateHeatsinkDamage(float damage)
{
	heatsinkDamage += damage;

	if (TESTING) {
		cout << "Heatsinks damaged by " << damage * 100 << " %" << endl;
	}
}

void CMechHeatTrack::ChangeEnvironment(ThermalEnvironment environment)
{
	this->environment = environment;
	
	// determine base heat dependig on current surroundings
	switch (environment) {
		case SPACE:
			baseCoolingFactor = 2.0f;
			break;
		case IN_WATER:
			baseCoolingFactor = 1.30f;
			break;
		case IN_HOT_AIR:
			baseCoolingFactor = 0.70f;
			break;
		case IN_COLD_AIR:
			baseCoolingFactor = 1.20f;
			break;
		case IN_TEMPARATE_AIR:
			baseCoolingFactor = 1.0f;
			break;
		default:
			baseCoolingFactor = 1.0f;
	}

	if (TESTING) {
		string env = GetEnvironmentName(environment);		
		cout << "Changing environment to: " << env << endl;;
	}
}

void CMechHeatTrack::AddHeat(double heat)
{
	additionalHeat += heat;

	if (TESTING)
		cout << "Adding heat: " << heat << endl;;
}

void CMechHeatTrack::UseCoolant(int liters)
{
	if (coolantInTank == 0) {				// tank is empty, return
		if (TESTING)
			cout << "No coolant left in the tank!" << endl;
		return;
	}
	else if (coolantInTank > liters) {	// enough coolant in tank
		additionalCooling += (double) -(liters * 2);
		coolantInTank -= liters;
	}
	else {								// not enough, but not entirely empty
		additionalCooling += (double) -(coolantInTank * 2);
		if (TESTING)
			liters = coolantInTank; // set to actual amount used for correct test display
		coolantInTank = 0;
	}

	if (TESTING) {
		cout << "Using " << liters << " liters of coolant..." << endl;
		cout << "Remaining coolant: " << coolantInTank << " liters" << endl;
	}
}

double CMechHeatTrack::GetRateOfHeatChange() 
{
	double newRate;
	int heatsinkMultiplier = 1;		// default
	if (typeOfHeatSink == CMechHeatTrack::DOUBLE) 
		heatsinkMultiplier = 2;

	// determine cooling rate factoring in the environment and damage to the heatsinks
	double coolingRate = ( baseCoolingFactor 
		* (additionalCooling 
			- (1 - heatsinkDamage) * numberOfHeatSinks * heatsinkMultiplier) );
	// determine total heat rate (positive: more heat, negative cooling down)
	newRate = additionalHeat + coolingRate;

	// reset add variables
	additionalCooling = 0;
	additionalHeat = 0;
	
	if (TESTING) {
		cout << "Cooling rate: " << coolingRate << endl;
		cout << "Heat rate: " << newRate << endl;
	}

	return newRate;
}

double CMechHeatTrack::GetHeat()
{
	return totalHeat;
}

bool CMechHeatTrack::GetOverheatStatus() 
{
	return overHeat;
}

bool CMechHeatTrack::GetCriticalStatus()
{
	return criticalTemp;
}

// helper methods
//-----------------------------------------------------------------

string CMechHeatTrack::GetEnvironmentName(ThermalEnvironment environment)
{
	string env;

	switch (environment) {
		case SPACE:
			env = "We're in space, wohoo!";
			break;
		case IN_WATER:
			env = "In water";
			break;
		case IN_HOT_AIR:
			env = "In hot air";
			break;
		case IN_COLD_AIR:
			env = "In cold air";
			break;
		case IN_TEMPARATE_AIR:
			env = "In temperate air";
			break;
		default:
			env = "unknown";
	}

	return env;
}

void CMechHeatTrack::ShowStatus() {
	string env = GetEnvironmentName(environment);

	cout << "Mech Status Report" << endl;
	cout << "Base Cooling Factor: " << baseCoolingFactor << endl;
	cout << "# of Heatsinks: " << numberOfHeatSinks << endl;

	if(typeOfHeatSink == SINGLE) {
		cout << "Heatsink Type: Single" << endl;
	}
	else {
		cout << "Heatsink Type: Double" << endl;
	}

	cout << "Heatsink Damage: " << heatsinkDamage << endl;
	cout << "Current Environment: " << env << endl;
	cout << "Remaining Coolant: " << coolantInTank << " liters" << endl;
	cout << "Rate of Heat Change: " << rateOfHeatChange << endl;
	cout << "Total Heat: " << totalHeat << endl;
	cout << "Is Overheated? " << (overHeat ? "yes" : "no") << endl;
	cout << "Critical Temperature? " << (criticalTemp? "yes" : "no") << endl << endl;
}
